@extends('site.base')
@section('titulo', 'Validar Celular - ' . env('EMPRESA'))

@section('content')
    <div class="relative w-full">
        <div class="pt-[64px]">
            <div class="relative w-[95%] max-h-[100%] md:w-4/6 lg:w-3/6 xl:w-[40%] 4xl:w-[25%] my-6 mx-auto">
                <div class="overflow-auto rounded-2xl shadow-lg flex flex-col w-full h-full bg-white">
                    <div class="h-full p-6">
                        <h1 class="text-primary-green font-bold text-2xl mb-3 mt-6">Confirme seu WhatsApp</h1>
                        <p class="text-gray-500 mb-6 text-sm">
                            Enviamos um <strong>código de 6 dígitos</strong> via WhatsApp. Digite-o abaixo para validar sua conta.
                        </p>

                        {{-- ALERTAS --}}
                        @if (session('status'))
                            <div class="alert alert-info">{{ session('status') }}</div>
                        @endif
                        @if (session('error'))
                            <div class="alert alert-danger">{{ session('error') }}</div>
                        @endif

                        <form id="otpPhoneForm" action="{{ route('validar.token.celular') }}" method="POST" novalidate>
                            @csrf
                            {{-- Celular --}}
                            <div class="mb-4">
                                <label class="mb-1 text-primary-green">Seu WhatsApp</label>
                                <input type="text" class="bg-green-light w-full h-11 px-4 rounded-full text-gray-400 text-base focus:outline-none" value="{{ Auth::user()->celular }}" disabled>
                            </div>

                            {{-- TOKEN (OTP) --}}
                            <div class="mb-6">
                                <label class="mb-1 text-primary-green">Token</label>
                                <div class="flex justify-between gap-2" id="otpGroup">
                                    @for ($i = 0; $i < 6; $i++)
                                        <input style="border-radius: 100px;" class="otp-input w-12 h-12 bg-green-light text-center text-xl font-bold rounded-lg focus:outline-none focus:bg-green-light/60" maxlength="1" inputmode="numeric"
                                            pattern="[0-9]*">
                                    @endfor
                                </div>
                                <input type="hidden" name="token" id="token" value="{{ old('token') }}">
                                @error('token')
                                    <small class="text-danger mt-1 block">{{ $message }}</small>
                                @enderror
                            </div>

                            {{-- Botão Validar --}}
                            <div class="px-6">
                                <div class="flex justify-center w-full mt-6">
                                    <button type="button" id="btnValidarPhone"
                                        class="bg-primary-green w-full flex hover:bg-primary-green/90 
                                           rounded-full px-4 py-[0.6rem] disabled:bg-disabled 
                                           disabled:cursor-not-allowed relative justify-center items-center">

                                        <!-- Texto -->
                                        <span id="btnValidarText" class="text-center text-white">Validar</span>

                                        <!-- Loader 3 dots -->
                                        <span id="btnValidarLoader" class="hidden flex gap-3 items-center justify-center">
                                            <span class="dot"></span>
                                            <span class="dot delay-150"></span>
                                            <span class="dot delay-300"></span>
                                        </span>
                                    </button>
                                </div>
                            </div>
                        </form>

                        <p class="text-center text-sm text-gray-500 mt-6">
                            Não consegue acessar este número?
                            <a href="{{ route('minha.conta') }}" class="text-primary-green font-semibold">Atualizar celular</a>
                            para receber um novo código.
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('js')
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
        const OTP_LENGTH = 6;

        function onlyDigits(v) {
            return (v || '').replace(/\D/g, '');
        }

        function getOtpValue() {
            return Array.from(document.querySelectorAll('#otpGroup .otp-input'))
                .map(i => (i.value || '').replace(/\D/g, ''))
                .join('')
                .slice(0, OTP_LENGTH);
        }

        function syncHidden() {
            document.getElementById('token').value = getOtpValue();
        }

        document.addEventListener('DOMContentLoaded', function() {
            const inputs = Array.from(document.querySelectorAll('#otpGroup .otp-input'));

            // Repopula se veio do old()
            const oldToken = document.getElementById('token').value || '';
            if (oldToken) {
                const digits = onlyDigits(oldToken).split('');
                digits.forEach((d, i) => {
                    if (inputs[i]) inputs[i].value = d;
                });
            }

            // Eventos para cada input
            inputs.forEach((inp, idx) => {
                inp.addEventListener('input', (e) => {
                    e.target.value = onlyDigits(e.target.value).slice(-1);
                    syncHidden();
                    if (e.target.value && idx < inputs.length - 1) {
                        inputs[idx + 1].focus();
                    }
                });
                inp.addEventListener('keydown', (e) => {
                    if (e.key === 'Backspace' && !inp.value && idx > 0) {
                        inputs[idx - 1].focus();
                        inputs[idx - 1].value = '';
                        syncHidden();
                    }
                });
                inp.addEventListener('paste', (e) => {
                    e.preventDefault();
                    const paste = (e.clipboardData || window.clipboardData).getData('text');
                    const digits = onlyDigits(paste).slice(0, OTP_LENGTH).split('');
                    digits.forEach((d, i) => {
                        if (inputs[i]) inputs[i].value = d;
                    });
                    syncHidden();
                    if (digits.length > 0 && inputs[digits.length - 1]) {
                        inputs[digits.length - 1].focus();
                    }
                });
            });

            // Botão Validar
            const btnValidar = document.getElementById("btnValidarPhone");
            const btnText = document.getElementById("btnValidarText");
            const btnLoader = document.getElementById("btnValidarLoader");

            if (btnValidar) {
                btnValidar.addEventListener("click", function() {
                    syncHidden();
                    const token = ($('#token').val() || '').trim();

                    if (token.length !== OTP_LENGTH) {
                        Swal.fire({
                            icon: 'warning',
                            title: 'Código incompleto',
                            text: 'Digite os 6 dígitos do token.'
                        });
                        return;
                    }

                    // Loader
                    btnValidar.disabled = true;
                    btnValidar.classList.remove("bg-primary-green", "hover:bg-primary-green/90");
                    btnValidar.style.background = "none";
                    btnText.classList.add("hidden");
                    btnLoader.classList.remove("hidden");

                    // Requisição
                    $.post("{{ route('validar.token.celular') }}", {
                            _token: '{{ csrf_token() }}',
                            token
                        })
                        .done(result => {
                            if (result === 'ok' || (result.status === 'ok')) {
                                btnValidar.classList.add("hidden");
                                Swal.fire({
                                    icon: 'success',
                                    title: 'Celular validado!'
                                }).then(() => window.location.href = "{{ route('cliente.ingressos') }}");
                            } else {
                                Swal.fire({
                                    icon: 'error',
                                    title: 'Token inválido',
                                    text: result.message || 'Token expirado. Tente novamente.'
                                });
                            }
                        })
                        .fail(() => {
                            Swal.fire({
                                icon: 'error',
                                title: 'Erro',
                                text: 'Não foi possível validar. Tente novamente.'
                            });
                        })
                        .always(() => {
                            btnValidar.disabled = false;
                            btnValidar.classList.add("bg-primary-green", "hover:bg-primary-green/90");
                            btnValidar.style.background = "";
                            btnText.classList.remove("hidden");
                            btnLoader.classList.add("hidden");
                        });
                });
            }
        });
    </script>
@endsection
