<?php

namespace App\Http\Controllers;

use App\Models\Boleto;
use App\Models\Evento;
use App\Models\EventoCupomDesconto;
use App\Models\EventoPacote;
use App\Models\ExtratoComissoes;
use App\Models\GetWay;
use App\Models\Ingresso;
use App\Models\Pedido;
use App\Models\Token;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Mail;
use Illuminate\Support\Facades\Log;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Validation\ValidationException;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\DB;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Milon\Barcode\DNS1D;
use Symfony\Contracts\EventDispatcher\Event;

class PedidoController extends Controller
{

    protected function sendPushNotification($expoPushToken, $title, $body, $data = [])
    {
        $pushController = app(PushController::class);
        $request = new Request([
            'expoPushToken' => $expoPushToken,
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ]);

        $pushController->sendNotification($request);
    }


    public function cart(Request $request)
    {
        $request->session()->forget(['carrinho', 'desconto', 'pagamento']);
        session()->save();

        // 0) Zera o carrinho atual
        $cupom = session()->get('cupom');

        // 1) Validação
        $data = $request->validate([
            'evento_id' => ['required', 'integer', 'exists:eventos,id'],
            'pacotes'   => ['required', 'array'],          // pacotes[ID] = qtd
            'pacotes.*' => ['nullable', 'integer', 'min:0']
        ]);

        // 2) Quantidades > 0
        $quantidades = collect($data['pacotes'])
            ->map(fn($q) => (int) $q)
            ->filter(fn($q) => $q > 0);

        if ($quantidades->isEmpty()) {
            return back()->withErrors(['pacotes' => 'Selecione ao menos 1 ingresso.'])->withInput();
        }

        // 3) Evento ativo
        $evento = Evento::where('id', $data['evento_id'])->where('status', 'Ativo')->first();
        if (!$evento) {
            return back()->withErrors(['evento_id' => 'Evento indisponível.']);
        }

        // 4) Carrega pacotes do evento
        $pacotes = EventoPacote::withCount(['ingressos' => fn($q) => $q->where('status', '1')])
            ->where('evento_id', $evento->id)
            ->whereIn('id', $quantidades->keys()->all())
            ->get()
            ->keyBy('id');

        if ($pacotes->count() !== $quantidades->keys()->count()) {
            return back()->withErrors(['pacotes' => 'Há pacotes inválidos para este evento.'])->withInput();
        }

        // 5) Disponibilidade e montagem
        $agora = Carbon::now('America/Bahia');

        $ingressos = [];
        $qtdTotal  = 0;

        $subtotalSemTaxaCts = 0;
        $taxasCts           = 0;
        $subtotalComTaxaCts = 0;

        foreach ($quantidades as $pacoteId => $qtdSelecionada) {
            $p = $pacotes[$pacoteId];

            $ini = Carbon::parse($p->data_inicio, 'America/Bahia');
            $fim = Carbon::parse($p->data_fim, 'America/Bahia');
            if ($p->status !== 'Ativo' || !$agora->between($ini, $fim)) {
                return back()->withErrors(['pacotes' => "O pacote {$p->nome} não está disponível no momento."])->withInput();
            }

            $qtdTotalPacote = (int) ($p->qtd ?? 0);
            $vendidos       = (int) ($p->ingressos_count ?? 0);
            $restantes      = max(0, $qtdTotalPacote - $vendidos);
            if ($restantes < $qtdSelecionada) {
                return back()->withErrors(['pacotes' => "Estoque insuficiente para {$p->nome}. Restantes: {$restantes}."])->withInput();
            }

            // 🔹 Caso o pacote tenha múltiplos ingressos (ex: casadinha = 2, trio = 3)
            $qtdIngressosPorPacote = (int) ($p->qtd_ingressos ?? 1); // campo no banco (ex: 1, 2 ou 3)
            $totalIngressosGerar = $qtdSelecionada * $qtdIngressosPorPacote;

            // 🔹 Divide o preço e taxas igualmente entre os ingressos do pacote
            $precoTotalCts = (int) ($p->preco ?? 0);
            $taxaTotalCts  = (int) round(($evento->taxa_servico ?? 0) / 100 * $precoTotalCts);

            $precoUnitarioCts = intval($precoTotalCts / $qtdIngressosPorPacote);
            $taxaUnitariaCts  = intval($taxaTotalCts / $qtdIngressosPorPacote);
            $valorFinalUnit   = $precoUnitarioCts + $taxaUnitariaCts;

            // 🔹 Monta os ingressos
            for ($i = 0; $i < $totalIngressosGerar; $i++) {
                $ingressos[] = [
                    'pacote_id'     => $p->id,
                    'pacote_nome'   => $p->nome,
                    'nome_completo' => null,
                    'email'         => null,
                    'celular'       => null,
                    'cpf'           => null,
                    'preco'         => $precoUnitarioCts,
                    'taxa'          => $taxaUnitariaCts,
                    'valor_final'   => $valorFinalUnit,
                ];
            }

            // 🔹 Acumuladores
            $qtdTotal           += $totalIngressosGerar;
            $subtotalSemTaxaCts += $precoUnitarioCts * $totalIngressosGerar;
            $taxasCts           += $taxaUnitariaCts * $totalIngressosGerar;
            $subtotalComTaxaCts += $valorFinalUnit * $totalIngressosGerar;
        }

        // 6) Desconto do cupom
        $descontoPct = (int) (session('cupom.desconto') ?? 0);

        $descontoSubtotal = $descontoPct > 0 ? intval($subtotalSemTaxaCts * ($descontoPct / 100)) : 0;
        $descontoTaxa     = $descontoPct > 0 ? intval($taxasCts * ($descontoPct / 100)) : 0;

        $subtotalComDesconto = $subtotalSemTaxaCts - $descontoSubtotal;
        $taxasComDesconto    = $taxasCts - $descontoTaxa;
        $totalFinalCts       = $subtotalComDesconto + $taxasComDesconto;

        // 7) Grava carrinho
        $novoCarrinho = [
            'evento_id'             => $evento->id,
            'qtd_itens'             => $qtdTotal,
            'dados_ingressos'       => $ingressos,
            'subtotal_sem_taxa_cts' => $subtotalSemTaxaCts,
            'taxas_cts'             => $taxasCts,
            'subtotal_com_taxa_cts' => $subtotalComTaxaCts,
            'taxa_servico_%'        => (float) ($evento->taxa_servico ?? 0),

            'cupom' => session('cupom') ? [
                'id'        => session('cupom.id'),
                'cupom'     => session('cupom.cupom'),
                'evento_id' => session('cupom.evento_id'),
                'situacao' => session('cupom.situacao'),
                'desconto'  => $descontoPct,
            ] : null,

            'desconto_subtotal_cts' => $descontoSubtotal,
            'desconto_taxa_cts'     => $descontoTaxa,
            'total_final_cts'       => $totalFinalCts,
        ];

        session()->put('carrinho', $novoCarrinho);

        // dd($ingressos);

        return view('site.cart', [
            'evento'    => $evento,
            'ingressos' => $ingressos,
            'carrinho'  => $novoCarrinho,
            'cupom'     => $cupom,
        ]);
    }


    public function DadosIngresso()
    {
        $carrinho   = session()->get('carrinho');
        $cupom      = session()->get('cupom');

        if (empty($carrinho['evento_id'])) {
            return redirect()->route('site.home');
        }

        $evento = Evento::with('CamposPersonalizados')->find($carrinho['evento_id']);
        return view('site.dados-ingressos', compact('carrinho'), ['evento' => $evento, 'cupom' => $cupom]);
    }


    public function CompraStore(Request $request)
    {
        // 1) Validação do payload
        $data = $request->validate([
            'evento_id' => ['required', 'integer', 'exists:eventos,id'],
            'pacotes'   => ['required', 'array'],          // formato: pacotes[ID] = qtd
            'pacotes.*' => ['nullable', 'integer', 'min:0']
        ]);

        // 2) Normaliza quantidades (remove nulos/zeros)
        $quantidades = collect($data['pacotes'])
            ->map(fn($q) => (int) $q)
            ->filter(fn($q) => $q > 0);

        if ($quantidades->isEmpty()) {
            return back()
                ->withErrors(['pacotes' => 'Selecione ao menos 1 ingresso.'])
                ->withInput();
        }

        // 3) Carrega evento disponível (status + tipo)
        $evento = Evento::query()
            ->where('id', $data['evento_id'])
            ->where('status', 'Ativo')
            ->where('tipo', 'Público')
            ->first();

        if (!$evento) {
            return back()->withErrors(['evento_id' => 'Evento indisponível.']);
        }

        $agora        = Carbon::now('America/Bahia');
        $taxaServicoP = (float) ($evento->taxa_servico ?? 0); // % sobre o valor com desconto
        $cupom        = session('cupom'); // ['evento_id','desconto'=>%] se houver

        // 4) Busca pacotes do evento solicitados
        $pacoteIds = $quantidades->keys()->all();

        $pacotes = EventoPacote::withCount(['ingressos' => function ($q) {
            $q->where('status', '1'); // já vendidos/confirmados
        }])
            ->where('evento_id', $evento->id)
            ->whereIn('id', $pacoteIds)
            ->get()
            ->keyBy('id');

        if ($pacotes->count() !== count($pacoteIds)) {
            return back()->withErrors(['pacotes' => 'Há ingressos inválidos para este evento.'])->withInput();
        }

        // 5) Monta carrinho — cálculos em CENTAVOS
        $cart               = [];
        $qtdTotal           = 0;
        $subtotalSemTaxaCts = 0; // soma dos preços com desconto (sem taxa)
        $subtotalComTaxaCts = 0; // soma final (com taxa)

        foreach ($quantidades as $pacoteId => $qtd) {
            /** @var \App\Models\EventoPacote $p */
            $p = $pacotes[$pacoteId];

            // Vigência/Status
            $ini = Carbon::parse($p->data_inicio, 'America/Bahia');
            $fim = Carbon::parse($p->data_fim, 'America/Bahia');
            if ($p->status !== 'Ativo' || !$agora->between($ini, $fim)) {
                return back()->withErrors(['pacotes' => "O pacote {$p->nome} não está disponível no momento."])->withInput();
            }

            // Estoque
            $qtdTotalPacote = (int) ($p->qtd ?? 0);
            $vendidos       = (int) ($p->ingressos_count ?? 0);
            $restantes      = max(0, $qtdTotalPacote - $vendidos);
            if ($restantes < $qtd) {
                return back()->withErrors(['pacotes' => "Estoque insuficiente para {$p->nome}. Restantes: {$restantes}."])->withInput();
            }

            // Preço base em centavos (assumindo que já está em centavos no banco)
            $precoBaseCts = (int) ($p->preco ?? 0);
            $precoDescCts = $precoBaseCts;

            // Cupom (%)
            if ($cupom && (int)($cupom['evento_id'] ?? 0) === (int)$evento->id) {
                $descontoP    = (float) ($cupom['desconto'] ?? 0);   // ex.: 10 => 10%
                $precoDescCts = (int) round($precoBaseCts * (1 - ($descontoP / 100)));
            }

            // Taxa de serviço (%)
            $taxaCts       = $taxaServicoP > 0 ? (int) round($precoDescCts * ($taxaServicoP / 100)) : 0;
            $unitTotalCts  = $precoDescCts + $taxaCts;
            $lineTotalCts  = $unitTotalCts * $qtd;

            $cart[] = [
                'pacote_id'          => $p->id,
                'pacote_nome'        => $p->nome,
                'qtd'                => $qtd,
                'preco_base_cts'     => $precoBaseCts,
                'preco_desc_cts'     => $precoDescCts,
                'taxa_servico_cts'   => $taxaCts,
                'unit_total_cts'     => $unitTotalCts,
                'line_total_cts'     => $lineTotalCts,
            ];

            $qtdTotal            += $qtd;
            $subtotalSemTaxaCts  += ($precoDescCts * $qtd);
            $subtotalComTaxaCts  += $lineTotalCts;
        }

        // 6) Taxas extras/juros (configure conforme sua regra)
        $taxaProcCts = 0;
        $jurosP      = 0.0; // exemplo: juros parcelamento/cartão
        $jurosCts    = (int) round($subtotalComTaxaCts * ($jurosP / 100));
        $totalCts    = $subtotalComTaxaCts + $taxaProcCts + $jurosCts;

        // 7) Guarda carrinho na sessão
        $dados = session()->put('carrinho', [
            'evento_id'              => $evento->id,
            'qtd_itens'              => $qtdTotal,
            'itens'                  => $cart,
            'subtotal_sem_taxa_cts'  => $subtotalSemTaxaCts,
            'subtotal_com_taxa_cts'  => $subtotalComTaxaCts,
            'taxa_servico_%'         => $taxaServicoP,
            'taxa_proc_cts'          => $taxaProcCts,
            'juros_%'                => $jurosP,
            'juros_cts'              => $jurosCts,
            'total_cts'              => $totalCts,
            'cupom'                  => $cupom ? [
                'id'        => $cupom['id'] ?? null,
                'evento_id' => $cupom['evento_id'] ?? null,
                'desconto'  => $cupom['desconto'] ?? 0,
            ] : null,
        ]);

        dd($dados);

        // 8) Redireciona ao checkout (middleware pode exigir login)
        // return redirect()->route('checkout.index');
    }

    public function IngressosCartUpdate(Request $request)
    {
        // Participantes vindos do formulário
        $participantes = $request->input('participantes', []);

        // Carrinho atual em sessão
        $carrinho = session('carrinho', []);

        if (!empty($carrinho['dados_ingressos'])) {
            foreach ($carrinho['dados_ingressos'] as $index => &$ingresso) {

                if (!isset($participantes[$index])) {
                    continue;
                }

                /* ================= CAMPOS FIXOS ================= */
                $ingresso['nome_completo'] = $participantes[$index]['nome'] ?? null;
                $ingresso['email']         = $participantes[$index]['email'] ?? null;
                $ingresso['cpf']           = $participantes[$index]['cpf'] ?? null;
                $ingresso['celular']       = $participantes[$index]['celular'] ?? null;
                $ingresso['genero']        = $participantes[$index]['genero'] ?? null;
                $ingresso['nascimento']    = $participantes[$index]['nascimento'] ?? null;

                /* ================= CAMPOS PERSONALIZADOS ================= */
                $camposExtras = collect($participantes[$index])
                    ->except(['nome', 'email', 'cpf', 'celular', 'genero', 'nascimento', 'foto'])
                    ->toArray();

                /* ================= FOTO ================= */
                if ($request->hasFile("participantes.$index.foto")) {

                    $file = $request->file("participantes.$index.foto");

                    if ($file->isValid()) {

                        $filename = 'ingresso-' . time() . '-' . $index . '.' . $file->getClientOriginalExtension();

                        $path = $file->storeAs('ingressos/fotos', $filename, 'public');

                        $camposExtras['foto'] = env('BASE_STORAGE') . $path;
                    }
                }

                // Salva os campos personalizados no ingresso da sessão
                $ingresso['campos_personalizados'] = $camposExtras;
            }
        }

        // Atualiza a sessão com os dados novos
        session(['carrinho' => $carrinho]);

        // Redireciona para o checkout
        return redirect()->route('site.checkout');
    }


    private function calcularTotaisOFF($carrinho, $evento, Request $request)
    {
        $totalPedido = $carrinho['subtotal_sem_taxa_cts'];
        $taxas       = $carrinho['taxas_cts'];
        $total       = $carrinho['subtotal_com_taxa_cts'];

        // Taxa de processamento e juros (somente cartão)
        $taxaProcessamento = $request->token ? 4.98 : 0;
        $juros = [
            1 => 0,
            2 => 2.53,
            3 => 4.62,
            4 => 6.69,
            5 => 8.66,
            6 => 9.96,
            7 => 11.24,
            8 => 12.50,
            9 => 13.73,
            10 => 14.93,
        ];

        $parcelas = (int) ($request->installments ?? 1);
        $jurosSelecionado = $juros[$parcelas] ?? 0;
        $totalJuros = $taxaProcessamento + $jurosSelecionado;

        // Recalcula se for cartão
        if ($request->payment_method_id !== 'pix') {
            $total = $total + porcentagem($totalJuros, $total);
        }

        return [
            'totalPedido'       => $totalPedido,
            'taxas'             => $taxas,
            'total'             => $total,
            'taxaProcessamento' => $taxaProcessamento,
            'juros'             => $jurosSelecionado,
            'totalJuros'        => $totalJuros,
            'parcelas'          => $parcelas,
        ];
    }


    public function pay(Request $request)
    {
        // dd($request->all());
        $carrinho = session()->get('carrinho');
        $getway   = session()->get('getway');
        $cupom    = $carrinho['cupom'] ?? null;
        $cupom    = $carrinho['cupom'] ?? null;
        $divulgador = session()->get('divulgador');


        if (!$carrinho || !$getway) {
            return back()->with('error', 'Sessão expirada. Refaça seu carrinho.');
        }

        $evento = Evento::find($carrinho['evento_id']);
        if (!$evento) {
            return back()->with('error', 'Evento não encontrado.');
        }

        // ===================== CÁLCULO DOS TOTAIS =====================
        $subtotalSemTaxaCts = (int) ($carrinho['subtotal_sem_taxa_cts'] ?? 0);
        $taxasCts           = (int) ($carrinho['taxas_cts'] ?? 0);
        $subtotalComTaxaCts = (int) ($carrinho['subtotal_com_taxa_cts'] ?? 0);

        // aplica desconto do cupom sobre o subtotal + taxa
        $descontoPct = (int) ($cupom['desconto'] ?? 0);
        $desconto    = $descontoPct > 0 ? porcentagem($descontoPct, $subtotalComTaxaCts) : 0;

        $totalFinalCts = $subtotalComTaxaCts - $desconto; // ✅ sempre em centavos
        $valorFinal    = moeda2($totalFinalCts);          // ✅ em reais (float/string)

        $fatorDesconto = $descontoPct > 0 ? (1 - ($descontoPct / 100)) : 1;

        // ===================== CRIA PEDIDO =====================
        do {
            $registro = gerarLetraRegistro(6);
        } while (Pedido::where('registro', $registro)->exists());

        $pedido = Pedido::create([
            'evento_id'       => $evento->id,
            'user_id'         => Auth::id(),
            'divulgador_id'   => $divulgador['id'] ?? 1,
            'taxa_origen'     => $evento->taxa_servico == 0 ? 'Empresa' : 'Cliente',
            'pdv'             => 'Site',
            'qtd_pacotes'     => $carrinho['qtd_itens'],
            'total'           => $totalFinalCts,                       // total final em centavos
            'sub_total'       => (int) round($subtotalSemTaxaCts * $fatorDesconto),
            'taxa'            => (int) round($taxasCts * $fatorDesconto),
            'forma_pagamento' => $request->payment_method_id === 'pix' ? 'PIX' : 'CC',
            'status'          => 'Pendente',
            'registro'        => $registro,
            'cupom'           => $cupom['cupom'] ?? null,
            'cupom_id'        => $cupom['id'] ?? null,
        ]);

        // ===================== CRIA BOLETO =====================
        $boleto = Boleto::create([
            'getway_id'   => $getway['id'],
            'pedido_id'   => $pedido->id,
            'user_id'     => Auth::id(),
            'status'      => '0',
            'valor'       => $totalFinalCts, // sempre em centavos
            'ref'         => $evento->titulo . ' (' . Auth::user()->name . ' ' . Auth::user()->sobrenome . ')',
            'vencimento'  => now()->addDays(2),
            'taxa_lopak'  => 50,
        ]);

        // ===================== CRIA INGRESSOS =====================
        foreach ($carrinho['dados_ingressos'] as $item) {
            do {
                $registroIngresso = gerarLetraRegistro(6);
            } while (Ingresso::where('registro', $registroIngresso)->exists());

            do {
                $codeBarras = gerarCodigo(6);
            } while (Ingresso::where('code_barras', $codeBarras)->exists());

            $precoComDesconto = (int) round($item['preco'] * $fatorDesconto);
            $taxaComDesconto  = (int) round($item['taxa'] * $fatorDesconto);
            $valorFinalIng    = $precoComDesconto + $taxaComDesconto;

            // 🔹 Campos personalizados no formato JSON
            $camposPersonalizados = !empty($item['campos_personalizados'])
                ? json_encode($item['campos_personalizados'], JSON_UNESCAPED_UNICODE)
                : null;

            Ingresso::create([
                'pedido_id'            => $pedido->id,
                'user_id'              => Auth::id(),
                'evento_id'            => $evento->id,
                'pacote_id'            => $item['pacote_id'],
                'pdv'                  => 'Site',
                'status'               => '0',
                'check_in_status'      => '0',
                'check_tentativas'     => '0',
                'code_barras'          => $codeBarras,
                'registro'             => $registroIngresso,
                'num_acessos'          => 0,
                'nome_completo'        => $item['nome_completo'],
                'email'                => $item['email'],
                'cpf'                  => $item['cpf'],
                'celular'              => $item['celular'],
                'preco'                => $precoComDesconto,
                'taxa'                 => $taxaComDesconto,
                'valor_final'          => $valorFinalIng,
                'cupom'                => $cupom['cupom'] ?? null,
                'cupom_id'             => $cupom['id'] ?? null,
                'campos_personalizados' => $camposPersonalizados, // ✅ aqui vai o JSON
            ]);
        }

        $items = [];
        foreach ($carrinho['dados_ingressos'] as $item) {
            $pacote = EventoPacote::find($item['pacote_id']);

            $precoComDesconto = (int) round($item['preco'] * $fatorDesconto);
            $taxaComDesconto  = (int) round($item['taxa'] * $fatorDesconto);
            $valorFinalIng    = $precoComDesconto + $taxaComDesconto;

            $items[] = [
                "id"          => (string) $item['pacote_id'],
                "title"       => $pacote->nome . " - " . $evento->titulo . ' (' . $item['nome_completo'] . ')',
                "description" => $pacote->nome . " - " . $evento->titulo,
                "quantity"    => 1,
                "unit_price"  => moeda2($valorFinalIng), // valor em reais
                "category_id" => "tickets",
            ];
        }

        // ===================== MERCADO PAGO =====================
        $date = Carbon::now()->addMinutes(10)->setTimezone('-03:00')->format("Y-m-d\TH:i:s.000P");
        $idempotencyKey = Str::uuid()->toString();

        if ($request->payment_method_id === 'pix') {
            $body = [
                "transaction_amount" => (float) $valorFinal, // ✅ convertido com moeda2
                "description"        => $carrinho['qtd_itens'] . ' ingresso(s) ' . $evento->titulo,
                "external_reference" => $boleto->id,
                "notification_url"   => route('webhook.mercadopago'),
                "payment_method_id"  => "pix",
                "date_of_expiration" => $date,
                "payer" => [
                    "first_name" => Auth::user()->name,
                    "last_name"  => Auth::user()->sobrenome,
                    "email"      => Auth::user()->email,
                    "identification" => [
                        "type"   => "CPF",
                        "number" => preg_replace('/\D/', '', Auth::user()->cpf)
                    ]
                ],
                "additional_info" => [
                    "items" => $items
                ]
            ];
        } else {
            $nomeCompleto = $request->name ?? (Auth::user()->name . ' ' . Auth::user()->sobrenome);
            $nomeSeparado = explode(' ', $nomeCompleto);
            $primeiroNome = $nomeSeparado[0];
            $sobreNome    = implode(' ', array_slice($nomeSeparado, 1));

            // aplica taxa 4,98%
            $valorFinalComTaxa = $totalFinalCts + (int) round(porcentagem(4.98, $totalFinalCts));
            $valorFinalComTaxa = moeda2($valorFinalComTaxa); // ✅ convertido só aqui

            $body = [
                "description"        => $carrinho['qtd_itens'] . ' ingresso(s) ' . $evento->titulo,
                "external_reference" => $boleto->id,
                "installments"       => (int) ($request->installments ?? 1),
                "notification_url"   => route('webhook.mercadopago'),
                "date_of_expiration" => $date,
                "payer" => [
                    "first_name" => $primeiroNome,
                    "last_name"  => $sobreNome,
                    "email"      => $request->payer['email'] ?? Auth::user()->email,
                    "identification" => [
                        "type"   => "CPF",
                        "number" => preg_replace('/\D/', '', $request->cpf ?? Auth::user()->cpf)
                    ]
                ],
                "payment_method_id" => $request->payment_method_id,
                "token"             => $request->token,
                "transaction_amount" => (float) $valorFinalComTaxa,
                "additional_info" => [
                    "items" => $items
                ]
            ];
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $getway['access_token'],
            'Content-Type'  => 'application/json',
            'X-Idempotency-Key' => $idempotencyKey,
        ])->post('https://api.mercadopago.com/v1/payments', $body);

        $dados     = $response->json();


        $paymentId = (string) data_get($dados, 'id');

        $boleto->update(['payment_id' => $paymentId]);

        return response()->json([
            'id'        => $paymentId,
            'pedido_id' => $pedido->id,
            'boleto_id' => $boleto->id,
            'status'    => $response->status(),
            'body'      => $dados,
        ]);
    }


    public function RetornoMercadoPago($id)
    {
        // $id recebido é o PAYMENT_ID do Mercado Pago
        $Boleto = Boleto::with(['Pedido', 'Pedido.Ingressos', 'Pedido.Evento'])
            ->where('payment_id', $id) // agora busca pelo payment_id salvo
            ->first();

        if (!$Boleto) {
            return response()->json([
                'success' => false,
                'message' => 'Pagamento não encontrado para o payment_id informado.',
            ], 404);
        }

        $pedidoId = $Boleto->pedido_id;
        $getway   = GetWay::find($Boleto->getway_id);
        if (!$getway || empty($getway->access_token)) {
            return response()->json([
                'success' => false,
                'message' => 'GetWay não encontrado ou access_token ausente.',
            ], 500);
        }

        $accessToken = $getway->access_token;

        try {
            // Consulta direta pelo payment_id
            $response = Http::withToken($accessToken)
                ->timeout(20)
                ->get("https://api.mercadopago.com/v1/payments/{$id}");

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status'  => $response->status(),
                    'body'    => $response->json(),
                ], $response->status());
            }

            $dados    = $response->json();
            $statusMp = data_get($dados, 'status'); // approved, pending, in_process, rejected, etc.

            // Mapa de conversão (Banco recebe número)
            $mapStatus = [
                'pending'      => 0,
                'in_process'   => 0,
                'authorized'   => 0,
                'approved'     => 1,
                'rejected'     => 2,
                'cancelled'    => 2,
                'refunded'     => 2,
                'charged_back' => 2,
            ];

            $statusNum = $mapStatus[$statusMp] ?? 0;

            // Atualiza no banco só o número
            // $Boleto->forceFill([
            //     'payment_id' => (string) $id,
            //     'status'     => $statusNum,
            // ])->save();

            $Pedido = $Boleto->Pedido;
            if (!$Pedido) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pedido não encontrado.',
                ], 404);
            }

            // Se aprovado
            if ($statusNum === 1) {
                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => now()->toDateString(),
                ]);
                $Pedido->update(['status' => 'Aprovado']);
                ExtratoComissoes::create([
                    'user_id'   => $Pedido->divulgador_id,
                    'pedido_id' => $Pedido->pedido_id,
                    'valor'     => porcentagem(env('COMISSAO'), $Pedido->total),
                    'status'    => '1'
                ]);
                $Divulgador = User::find($Pedido->divulgador_id);
                $Divulgador->update(['saldo' => $Divulgador->saldo + porcentagem(env('COMISSAO'), $Pedido->total)]);

                $ultimoUsuario = null;
                foreach ($Pedido->Ingressos as $ingresso) {
                    $ingresso->update(['status' => '1']);

                    // RULE OFF
                    if (!empty($ingresso->Evento->rule_off) && ($getway->tipo ?? null) === 'pf') {
                        $ingresso->update(['check_in_situacao' => '1']);
                        $Pedido->update(['check_in_situacao' => '1']);
                        foreach ($Pedido->Ingressos as $ddx) {
                            $ddx->update(['check_in_situacao' => '1']);
                        }
                    }

                    $user = User::find($ingresso->user_id);
                    $ultimoUsuario = $user ?: $ultimoUsuario;

                    if ($user) {
                        try {
                            Mail::send('emails.usuario-pdv', [
                                'name'   => $user->name,
                                'email'  => $user->email,
                                'evento' => $Pedido->Evento->titulo,
                                'banner' => $Pedido->Evento->banner,
                                'link'   => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                                'qtd'    => $Pedido->Ingressos->count(),
                                'ref'    => $Pedido->Ingressos->count() . ' ingresso(s) ' . $Pedido->Evento->titulo,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de ingresso', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                // Push notification (se existir)
                if ($ultimoUsuario && method_exists($this, 'sendPushNotification')) {
                    foreach ($ultimoUsuario->pushToken as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                'Ingresso Disponível',
                                'Seu ingresso para ' . $Pedido->Evento->titulo . ' já está disponível.',
                                ['evento_id' => $Pedido->Evento->id ?? 1]
                            );
                        }
                    }
                }

                return response()->json([
                    'success'  => true,
                    'response' => $statusMp, // ← Texto do Mercado Pago
                    'dados'    => $dados,
                    'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
                    'redirect' => 'https://checkout.ucesub.com.br/success/' . $Pedido->id,
                ]);
            }

            // ====================== CANCELADO ======================
            if ($statusNum === 2) {
                $Boleto->update(['status'   => '2']);
                $Pedido->update(['status'   => 'Cancelado']);

                foreach ($Pedido->Ingressos as $ingresso) {
                    $ingresso->update(['status' => '2']);
                    $user = User::find($ingresso->user_id);
                    if ($user) {
                        try {
                            Mail::send('emails.cancelamento-de-compra', [
                                'name'     => $user->name,
                                'evento'   => $Pedido->Evento->titulo,
                                'banner'   => $Pedido->Evento->banner,
                                'link'     => route('site.evento', [$Pedido->Evento->url, $Pedido->Evento->id]),
                                'ingresso' => $ingresso,
                                'pedido'   => $Pedido,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Cancelamento de Ingresso - ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de cancelamento', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                return response()->json([
                    'success'  => true,
                    'response' => $statusMp,
                    'dados'    => $dados,
                    'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
                    'redirect' => 'https://checkout.ucesub.com.br/cancel/' . $Pedido->id,
                ]);
            }

            // Retorna outros status em TEXTO
            return response()->json([
                'success'  => true,
                'response' => $statusMp ?: 'unknown',
                'dados'    => $dados,
                'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }


    public function WebhookMercadoPago(Request $request)
    {
        $id = $request->get('id');
        $topic = $request->get('topic'); // geralmente será "payment"

        if (!$id || $topic !== 'payment') {
            return response()->json(['success' => false, 'message' => 'Notificação inválida.'], 400);
        }
        // $id recebido é o PAYMENT_ID do Mercado Pago
        $Boleto = Boleto::with(['Pedido', 'Pedido.Ingressos', 'Pedido.Evento'])
            ->where('payment_id', $id) // agora busca pelo payment_id salvo
            ->first();

        if (!$Boleto) {
            return response()->json([
                'success' => false,
                'message' => 'Pagamento não encontrado para o payment_id informado.',
            ], 404);
        }

        $pedidoId = $Boleto->pedido_id;
        $getway   = GetWay::find($Boleto->getway_id);
        if (!$getway || empty($getway->access_token)) {
            return response()->json([
                'success' => false,
                'message' => 'GetWay não encontrado ou access_token ausente.',
            ], 500);
        }

        $accessToken = $getway->access_token;

        try {
            // Consulta direta pelo payment_id
            $response = Http::withToken($accessToken)
                ->timeout(20)
                ->get("https://api.mercadopago.com/v1/payments/{$id}");

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status'  => $response->status(),
                    'body'    => $response->json(),
                ], $response->status());
            }

            $dados    = $response->json();
            $statusMp = data_get($dados, 'status'); // approved, pending, in_process, rejected, etc.

            // Mapa de conversão (Banco recebe número)
            $mapStatus = [
                'pending'      => 0,
                'in_process'   => 0,
                'authorized'   => 0,
                'approved'     => 1,
                'rejected'     => 2,
                'cancelled'    => 2,
                'refunded'     => 2,
                'charged_back' => 2,
            ];

            $statusNum = $mapStatus[$statusMp] ?? 0;

            // Atualiza no banco só o número
            // $Boleto->forceFill([
            //     'payment_id' => (string) $id,
            //     'status'     => $statusNum,
            // ])->save();

            $Pedido = $Boleto->Pedido;
            if (!$Pedido) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pedido não encontrado.',
                ], 404);
            }

            // Se aprovado
            if ($statusNum === 1) {
                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => now()->toDateString(),
                ]);

                $Pedido->update(['status' => 'Aprovado']);

                $ultimoUsuario = null;
                foreach ($Pedido->Ingressos as $ingresso) {
                    $ingresso->update(['status' => '1']);

                    // RULE OFF
                    if (!empty($ingresso->Evento->rule_off) && ($getway->tipo ?? null) === 'pf') {
                        $ingresso->update(['check_in_situacao' => '1']);
                        $Pedido->update(['check_in_situacao' => '1']);
                        foreach ($Pedido->Ingressos as $ddx) {
                            $ddx->update(['check_in_situacao' => '1']);
                        }
                    }

                    $user = User::find($ingresso->user_id);
                    $ultimoUsuario = $user ?: $ultimoUsuario;

                    if ($user) {
                        try {
                            Mail::send('emails.usuario-pdv', [
                                'name'   => $user->name,
                                'email'  => $user->email,
                                'evento' => $Pedido->Evento->titulo,
                                'banner' => $Pedido->Evento->banner,
                                'link'   => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                                'qtd'    => $Pedido->Ingressos->count(),
                                'ref'    => $Pedido->Ingressos->count() . ' ingresso(s) ' . $Pedido->Evento->titulo,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de ingresso', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                // Push notification (se existir)
                if ($ultimoUsuario && method_exists($this, 'sendPushNotification')) {
                    foreach ($ultimoUsuario->pushToken as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                'Ingresso Disponível',
                                'Seu ingresso para ' . $Pedido->Evento->titulo . ' já está disponível.',
                                ['evento_id' => $Pedido->Evento->id ?? 1]
                            );
                        }
                    }
                }

                return response()->json([
                    'success'  => true,
                    'response' => $statusMp, // ← Texto do Mercado Pago
                    'dados'    => $dados,
                    'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
                    'redirect' => 'https://checkout.ucesub.com.br/success/' . $Pedido->id,
                ]);
            }

            // ====================== CANCELADO ======================
            if ($statusNum === 2) {
                $Boleto->update(['status'   => '2']);
                $Pedido->update(['status'   => 'Cancelado']);

                foreach ($Pedido->Ingressos as $ingresso) {
                    $ingresso->update(['status' => '2']);
                    $user = User::find($ingresso->user_id);
                    if ($user) {
                        try {
                            Mail::send('emails.cancelamento-de-compra', [
                                'name'     => $user->name,
                                'evento'   => $Pedido->Evento->titulo,
                                'banner'   => $Pedido->Evento->banner,
                                'link'     => route('site.evento', [$Pedido->Evento->url, $Pedido->Evento->id]),
                                'ingresso' => $ingresso,
                                'pedido'   => $Pedido,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Cancelamento de Ingresso - ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de cancelamento', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                return response()->json([
                    'success'  => true,
                    'response' => $statusMp,
                    'dados'    => $dados,
                    'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
                    'redirect' => 'https://checkout.ucesub.com.br/cancel/' . $Pedido->id,
                ]);
            }

            // Retorna outros status em TEXTO
            return response()->json([
                'success'  => true,
                'response' => $statusMp ?: 'unknown',
                'dados'    => $dados,
                'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
            ]);
        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function index()
    {
        $userId = Auth::user()->id;
        $Pedidos = Pedido::where('user_id', $userId)->get();
        foreach ($Pedidos as $pedido) {
            if ($pedido->status === 'Pendente') {
                $boleto = Boleto::where('pedido_id', $pedido->id)->first();
                if ($boleto && !empty($boleto->payment_id)) {
                    $this->ConsultarPagamento($boleto->payment_id);
                }
                $Pedidos = Pedido::where('user_id', $userId)->get();
                return view('cliente.meus-ingressos', compact('Pedidos'));
            }
        }
        return view('cliente.meus-ingressos', compact('Pedidos'));
    }

    public function ConsultarPagamento($id)
    {
        // $id recebido é o PAYMENT_ID do Mercado Pago
        $Boleto = Boleto::with(['Pedido', 'Pedido.Ingressos', 'Pedido.Evento'])
            ->where('payment_id', $id) // agora busca pelo payment_id salvo
            ->first();


        if (!$Boleto) {
            return response()->json([
                'success' => false,
                'message' => 'Pagamento não encontrado para o payment_id informado.',
            ], 404);
        }

        $pedidoId = $Boleto->pedido_id;
        $getway   = GetWay::find($Boleto->getway_id);
        if (!$getway || empty($getway->access_token)) {
            return response()->json([
                'success' => false,
                'message' => 'GetWay não encontrado ou access_token ausente.',
            ], 500);
        }

        $accessToken = $getway->access_token;

        try {
            // Consulta direta pelo payment_id
            $response = Http::withToken($accessToken)
                ->timeout(20)
                ->get("https://api.mercadopago.com/v1/payments/{$id}");

            if ($response->failed()) {
                return redirect()->back()->with('error', 'Falha ao consultar pagamento.');
            }

            $dados    = $response->json();
            $statusMp = data_get($dados, 'status'); // approved, pending, in_process, rejected, etc.

            // Mapa de conversão (Banco recebe número)
            $mapStatus = [
                'pending'      => 0,
                'in_process'   => 0,
                'authorized'   => 0,
                'approved'     => 1,
                'rejected'     => 2,
                'cancelled'    => 2,
                'refunded'     => 2,
                'charged_back' => 2,
            ];

            $statusNum = $mapStatus[$statusMp] ?? 0;

            $Pedido = $Boleto->Pedido;
            if (!$Pedido) {
                return redirect()->back()->with('error', 'Pedido não encontrado.');
            }

            // Se aprovado
            if ($statusNum === 1) {
                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => now()->toDateString(),
                ]);

                $Pedido->update(['status' => 'Aprovado']);

                $ultimoUsuario = null;
                foreach ($Pedido->Ingressos as $ingresso) {
                    $ingresso->update(['status' => '1']);

                    // RULE OFF
                    if (!empty($ingresso->Evento->rule_off) && ($getway->tipo ?? null) === 'pf') {
                        $ingresso->update(['check_in_situacao' => '1']);
                        $Pedido->update(['check_in_situacao' => '1']);
                        foreach ($Pedido->Ingressos as $ddx) {
                            $ddx->update(['check_in_situacao' => '1']);
                        }
                    }

                    $user = User::find($ingresso->user_id);
                    $ultimoUsuario = $user ?: $ultimoUsuario;

                    if ($user) {
                        try {
                            Mail::send('emails.usuario-pdv', [
                                'name'   => $user->name,
                                'email'  => $user->email,
                                'evento' => $Pedido->Evento->titulo,
                                'banner' => $Pedido->Evento->banner,
                                'link'   => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                                'qtd'    => $Pedido->Ingressos->count(),
                                'ref'    => $Pedido->Ingressos->count() . ' ingresso(s) ' . $Pedido->Evento->titulo,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de ingresso', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                // ====================== CANCELADO ======================
                if ($statusNum === 2) {
                    $Boleto->update(['status'   => '2']);
                    $Pedido->update(['status'   => 'Cancelado']);

                    foreach ($Pedido->Ingressos as $ingresso) {
                        $ingresso->update(['status' => '2']);
                        $user = User::find($ingresso->user_id);
                        if ($user) {
                            try {
                                Mail::send('emails.cancelamento-de-compra', [
                                    'name'     => $user->name,
                                    'evento'   => $Pedido->Evento->titulo,
                                    'banner'   => $Pedido->Evento->banner,
                                    'link'     => route('site.evento', [$Pedido->Evento->url, $Pedido->Evento->id]),
                                    'ingresso' => $ingresso,
                                    'pedido'   => $Pedido,
                                ], function ($mail) use ($user) {
                                    $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                    $mail->to($user->email)->subject('Cancelamento de Ingresso - ' . env('APP_NAME'));
                                });
                            } catch (\Throwable $e) {
                                Log::warning('Falha ao enviar e-mail de cancelamento', [
                                    'pedido_id' => $Pedido->id,
                                    'user_id'   => $user->id ?? null,
                                    'erro'      => $e->getMessage(),
                                ]);
                            }
                        }
                    }

                    return response()->json([
                        'success'  => true,
                        'response' => $statusMp,
                        'dados'    => $dados,
                        'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
                        'redirect' => 'https://checkout.ucesub.com.br/cancel/' . $Pedido->id,
                    ]);
                }

                // Push notification (se existir)
                if ($ultimoUsuario && method_exists($this, 'sendPushNotification')) {
                    foreach ($ultimoUsuario->pushToken as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                'Ingresso Disponível',
                                'Seu ingresso para ' . $Pedido->Evento->titulo . ' já está disponível.',
                                ['evento_id' => $Pedido->Evento->id ?? 1]
                            );
                        }
                    }
                }
                return redirect()->back()->with('success', 'Pagamento confirmado com sucesso.');
            } else {
                return redirect()->back()->with('error', 'Seu pagamento ainda se encontra pendente.');
            }
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', 'Erro ao consultar pagamento.');
        }
    }


    public function checkout()
    {
        $carrinho = session()->get('carrinho');
        $evento   = Evento::find($carrinho['evento_id']);
        $user     = Auth::user();

        $fotoUsuario = null;

        if (!empty($carrinho['dados_ingressos'][0]['campos_personalizados']['foto'])) {
            $fotoUsuario = $carrinho['dados_ingressos'][0]['campos_personalizados']['foto'];
        }

        if ($fotoUsuario && $user->foto !== $fotoUsuario) {
            $user->update([
                'foto' => $fotoUsuario
            ]);
        }

        // ========================================================
        // 🔒 VERIFICA SE JÁ EXISTE PEDIDO PENDENTE
        // ========================================================
        $pedidoPendente = Pedido::where('evento_id', $evento->id)
            ->where('user_id', $user->id)
            ->where('status', 'Pendente')
            ->latest()
            ->first();

        if ($pedidoPendente) {

            $boleto = Boleto::where('pedido_id', $pedidoPendente->id)->first();

            if ($boleto) {
                return view('site.pix-asaas', [
                    'Pedido' => $pedidoPendente,
                    'boleto' => $boleto
                ]);
            }

            // Segurança extra: se por algum motivo não existir boleto
            return redirect()
                ->route('site.evento', [$evento->url, $evento->id])
                ->with('warning', 'Você já possui um pedido pendente para este evento.');
        }

        // ========================================================
        // 🆓 CASO INGRESSO GRATUITO
        // ========================================================
        if ($carrinho['total_final_cts'] == 0 || $carrinho['cupom']['situacao'] == '1') {
            do {
                $registro = gerarLetraRegistro(6);
            } while (Pedido::where('registro', $registro)->exists());

            $pedido = Pedido::create([
                'evento_id'       => $evento->id,
                'user_id'         => $user->id,
                'taxa_origen'     => 'Gratuito',
                'pdv'             => 'Site',
                'qtd_pacotes'     => $carrinho['qtd_itens'],
                'total'           => 0,
                'sub_total'       => 0,
                'taxa'            => 0,
                'forma_pagamento' => 'Gratuito',
                'status'          => 'Aprovado',
                'registro'        => $registro,
                'cupom'           => $carrinho['cupom']['cupom'] ?? null,
                'cupom_id'        => $carrinho['cupom']['id'] ?? null,
            ]);

            foreach ($carrinho['dados_ingressos'] as $item) {
                do {
                    $registroIngresso = gerarLetraRegistro(6);
                } while (Ingresso::where('registro', $registroIngresso)->exists());

                do {
                    $codeBarras = gerarCodigo(6);
                } while (Ingresso::where('code_barras', $codeBarras)->exists());
                $camposPersonalizados = $item['campos_personalizados'] ?? null;

                Ingresso::create([
                    'pedido_id'             => $pedido->id,
                    'user_id'               => $user->id,
                    'evento_id'             => $evento->id,
                    'pacote_id'             => $item['pacote_id'],
                    'pdv'                   => 'Site',
                    'status'                => '1',
                    'check_in_status'       => '0',
                    'check_tentativas'      => '0',
                    'code_barras'           => $codeBarras,
                    'registro'              => $registroIngresso,
                    'num_acessos'           => 0,
                    'nome_completo'         => $item['nome_completo'],
                    'email'                 => $item['email'],
                    'cpf'                   => $item['cpf'],
                    'celular'               => $item['celular'],
                    'preco'                 => 0,
                    'taxa'                  => 0,
                    'valor_final'           => 0,
                    'cupom'                 => $carrinho['cupom']['cupom'] ?? null,
                    'cupom_id'              => $carrinho['cupom']['id'] ?? null,
                    'campos_personalizados' => $item['campos_personalizados'] ?? null,
                ]);
            }

            session()->forget('carrinho');
            return redirect()->route('cliente.ingressos')->with('success', 'Ingresso gratuito gerado com sucesso!');
        }

        // ========================================================
        // 💰 CASO NORMAL (PAGO) → Lógica ASAAS
        // ========================================================

        // Primeiro calcula totais
        $totais = $this->calcularTotais($carrinho, $evento, request());

        // Gera registro do pedido
        do {
            $registro = gerarLetraRegistro(6);
        } while (Pedido::where('registro', $registro)->exists());

        // Cria Pedido
        $pedido = Pedido::create([
            'evento_id'       => $evento->id,
            'user_id'         => $user->id,
            'tipo'            => 'evento',
            'taxa_origen'     => $evento->taxa_servico == 0 ? 'Empresa' : 'Cliente',
            'pdv'             => 'Site',
            'qtd_pacotes'     => $carrinho['qtd_itens'],
            'total'           => $totais['total'],
            'sub_total'       => $totais['totalPedido'],
            'taxa'            => $totais['taxas'],
            'forma_pagamento' => 'PIX',
            'status'          => 'Pendente',
            'registro'        => $registro,
            'cupom'           => $carrinho['cupom']['cupom'] ?? '',
            'cupom_id'        => $carrinho['cupom']['id'] ?? null,
        ]);

        // Cria Boleto / fatura local antes do ASAAS
        $boleto = Boleto::create([
            'pedido_id'   => $pedido->id,
            'user_id'     => $user->id,
            'status'      => '0',
            'valor'       => $totais['total'],
            'ref'         => $carrinho['dados_ingressos'][0]['pacote_nome'] . ' ' . $evento->titulo,
            'vencimento'  => now()->addDays(2)->format('Y-m-d'),
        ]);

        // Cria ingressos
        foreach ($carrinho['dados_ingressos'] as $item) {

            do {
                $registroIngresso = gerarLetraRegistro(6);
            } while (Ingresso::where('registro', $registroIngresso)->exists());

            do {
                $codeBarras = gerarCodigo(6);
            } while (Ingresso::where('code_barras', $codeBarras)->exists());

            Ingresso::create([
                'pedido_id'        => $pedido->id,
                'user_id'          => $user->id,
                'evento_id'        => $evento->id,
                'pacote_id'        => $item['pacote_id'],
                'pdv'              => 'Site',
                'status'           => '0',
                'check_in_status'  => '0',
                'check_tentativas' => '0',
                'code_barras'      => $codeBarras,
                'registro'         => $registroIngresso,
                'num_acessos'      => 0,
                'nome_completo'    => $item['nome_completo'],
                'email'            => $item['email'],
                'cpf'              => $item['cpf'],
                'celular'          => $item['celular'] ?? null,
                'preco'            => $item['preco'],
                'taxa'             => $item['taxa'],
                'valor_final'      => $item['valor_final'],
                'cupom'            => $carrinho['cupom']['cupom'] ?? '',
                'cupom_id'         => $carrinho['cupom']['id'] ?? null,
                'campos_personalizados' => $item['campos_personalizados'] ?? null,
            ]);
        }


        // ========================================================
        // 🔗 INTEGRAÇÃO ASAAS (exatamente igual ao checkoutAsaas)
        // ========================================================

        $client    = new Client();
        $asaasBase = env('ASAAS_PRODUCAO');
        $asaasKey  = env('KEYASAASA');

        // 1️⃣ Garantir CUSTOMER
        try {
            $clienteAsaasID = $this->ensureAsaasCustomer($user, $client, $asaasBase, $asaasKey);
        } catch (\Throwable $e) {
            // return back()->with('error', 'Erro nos dados pessoais: ' . $e->getMessage());
            return $e->getMessage();
        }

        // 2️⃣ Criar pagamento PIX no ASAAS
        try {
            $resFatura = $this->createAsaasPixPayment($client, $asaasBase, $asaasKey, $clienteAsaasID, $boleto);
        } catch (ClientException $e) {
            $body = json_decode($e->getResponse()->getBody()->getContents() ?? '{}', true);
            $code = $body['errors'][0]['code'] ?? null;

            if ($code === 'invalid_customer') {
                $user->update(['asaas_id' => null]);
                $clienteAsaasID = $this->ensureAsaasCustomer($user, $client, $asaasBase, $asaasKey);
                $resFatura = $this->createAsaasPixPayment($client, $asaasBase, $asaasKey, $clienteAsaasID, $boleto);
            } else {
                Log::error(' ASAAS PIX error ', ['body' => $body]);
                throw $e;
            }
        }

        // 3️⃣ Obter QR CODE PIX
        $responsePix = $client->get("$asaasBase/v3/payments/{$resFatura['id']}/pixQrCode", [
            'headers' => [
                'accept'       => 'application/json',
                'access_token' => $asaasKey,
            ],
        ]);

        $pix = json_decode($responsePix->getBody(), true);

        // Atualiza boleto local
        $boleto->update([
            'asaas_id'       => $resFatura['id'],
            'asaas_checkout' => $resFatura['invoiceUrl'] ?? null,
            'asaas_boleto'   => $resFatura['bankSlipUrl'] ?? null,
            'qr_code'        => $pix['encodedImage'] ?? null,
            'cod_pix'        => $pix['payload'] ?? null,
        ]);

        // Redireciona para tela PIX
        return view('site.pix-asaas', [
            'Pedido' => $pedido,
            'boleto' => $boleto
        ]);
    }




    /**
     * Garante um customer válido no Asaas PARA A SUA CONTA:
     * - valida o asaas_id atual (GET /customers/{id})
     * - se inválido/404, zera
     * - tenta localizar por CPF/CNPJ (ou e-mail)
     * - se não existir, cria e salva o id
     */
    private function ensureAsaasCustomer(User $user, Client $client, string $asaasBase, string $asaasKey): string
    {
        // Normaliza campos
        $cpf    = $this->onlyDigits($user->cpf);
        $email  = trim($user->email);
        $phone  = $this->onlyDigits($user->celular);
        $nome   = trim($user->name . ' ' . $user->sobrenome);

        // 🔎 Validações obrigatórias ANTES de ir pro Asaas
        $erros = [];
        if (!$cpf || !in_array(strlen($cpf), [11, 14])) {
            $erros['cpf'] = 'Informe um CPF ou CNPJ válido.';
        }
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $erros['email'] = 'Informe um e-mail válido.';
        }
        if (!$phone || strlen($phone) < 10) {
            $erros['celular'] = 'Informe um celular válido (DDD + número).';
        }
        if (empty(trim($nome))) {
            $erros['name'] = 'Informe seu nome completo.';
        }

        if (!empty($erros)) {
            $user->update(['asaas_id' => null]); // limpa qualquer id inválido
            throw ValidationException::withMessages($erros)
                ->redirectTo(route('minha.conta'));
        }

        // 1) Se já tem asaas_id, valida na API
        if (!empty($user->asaas_id)) {
            try {
                $resp = $client->get("$asaasBase/v3/customers/{$user->asaas_id}", [
                    'headers' => [
                        'accept'       => 'application/json',
                        'access_token' => $asaasKey,
                    ],
                ]);
                $data = json_decode($resp->getBody(), true);
                if (!empty($data['id'])) {
                    return $data['id']; // válido nesta conta
                }
            } catch (ClientException $e) {
                // 404 ou inválido → limpa e segue fluxo
                $user->update(['asaas_id' => null]);
            }
        }

        // 2) Busca por CPF no Asaas
        try {
            $resp = $client->get("$asaasBase/v3/customers", [
                'headers' => [
                    'accept'       => 'application/json',
                    'access_token' => $asaasKey,
                ],
                'query' => ['cpfCnpj' => $cpf],
            ]);
            $found = json_decode($resp->getBody(), true);
            if (($found['totalCount'] ?? 0) > 0) {
                $id = $found['data'][0]['id'];
                $user->update(['asaas_id' => $id]);
                return $id;
            }
        } catch (\Throwable $th) {
            Log::warning('ASAAS find customer error', ['e' => $th->getMessage()]);
        }

        // 3) Se não achou → cria novo
        $respCreate = $client->post("$asaasBase/v3/customers", [
            'headers' => [
                'accept'       => 'application/json',
                'access_token' => $asaasKey,
                'content-type' => 'application/json',
            ],
            'json' => [
                'name'                 => $nome,
                'cpfCnpj'              => $cpf,
                'email'                => $email,
                'mobilePhone'          => $phone,
                'externalReference'    => $user->id,
                'notificationDisabled' => true,
            ],
        ]);
        $created = json_decode($respCreate->getBody(), true);
        $user->update(['asaas_id' => $created['id']]);

        return $created['id'];
    }


    /**
     * Cria pagamento PIX no Asaas (NÃO chama duas vezes).
     */
    private function createAsaasPixPayment(Client $client, string $asaasBase, string $asaasKey, string $customerId, Boleto $boleto): array
    {
        $resp = $client->post("$asaasBase/v3/payments", [
            'headers' => [
                'accept'       => 'application/json',
                'access_token' => $asaasKey,
                'content-type' => 'application/json',
            ],
            'json' => [
                'billingType'       => 'PIX',
                'customer'          => $customerId,
                'value'             => moeda2($boleto->valor), // R$ (não centavos)
                'dueDate'           => $boleto->vencimento,
                'description'       => $boleto->ref,
                'externalReference' => $boleto->id,
            ],
        ]);

        return json_decode($resp->getBody(), true);
    }

    private function onlyDigits(?string $v): string
    {
        return preg_replace('/\D+/', '', (string)$v);
    }

    private function calcularTotais($carrinho, $evento, $request)
    {
        // já vem pronto no carrinho
        $subtotalComTaxa = $carrinho['subtotal_com_taxa_cts'] ?? 0;
        $subtotalSemTaxa = $carrinho['subtotal_sem_taxa_cts'] ?? 0;
        $taxasCarrinho   = $carrinho['taxas_cts'] ?? 0;

        // desconto (se houver cupom)
        $desconto = 0;
        if (!empty($carrinho['cupom'])) {
            $desconto = $carrinho['cupom']['valor'] ?? 0;
        }

        // se já aplicou taxa_servico no subtotal_com_taxa_cts, não soma de novo
        $taxaServico = 0;

        $total = $subtotalComTaxa - $desconto;

        return [
            'total'       => $total,               // total final
            'totalPedido' => $subtotalSemTaxa,     // só ingressos sem taxa
            'taxas'       => $taxasCarrinho,       // só taxas
        ];
    }

    public function novos()
    {
        $carteiras = Ingresso::whereIN('pacote_id', [578])->get();
        return view('admin.pedidos.index', compact('carteiras'));
    }

    public function print(Pedido $Pedido)
    {
        // Buscar apenas 1 ingresso do pedido
        $carteira = Ingresso::where('pedido_id', $Pedido->id)
            ->orderBy('id', 'asc')
            ->first();

        if (!$carteira) {
            return view('site.404');
        }

        /**
         * 🔹 Gerar QR Code NO MESMO PADRÃO
         * (arquivo físico, 77px, reaproveitável)
         */
        if (!is_file(public_path('storage/qrcode/' . $carteira->Pedido->registro . '.png'))) {
            QrCode::format('png')
                ->size(77)
                ->generate(
                    route('validar.carteira', $carteira->Pedido->registro),
                    public_path('storage/qrcode/' . $carteira->Pedido->registro . '.png')
                );
        }

        $carteira->qr_code = url('storage/qrcode/' . $carteira->Pedido->registro . '.png');

        /**
         * 🔹 Código de Barras (mantido)
         */
        $barcode = new DNS1D();
        $barcode->setStorPath(__DIR__ . '/cache/');

        $carteira->barcode_img = 'data:image/png;base64,' .
            $barcode->getBarcodePNG($carteira->Pedido->registro, 'C39');

        /**
         * 🔹 Enviar para view
         */
        return view('admin.pedidos.print', compact('carteira'));
    }



    public function gerarToken($id)
    {
        $token = Str::random(40); // token que vai na URL
        $evento = Evento::find($id);

        $cupom = EventoCupomDesconto::create([
            'evento_id'         => $id,
            'cupom'             => hash('sha256', $token),
            'desconto'          => 0,
            'limite_ingressos'  => 1,
            'status'            => '1',
            'situacao'          => 1
        ]);

        return route('site.evento', [$evento->url, $id, $cupom->cupom]);
    }


    public function ValidarCarteira(Request $request, $cod = null)
    {

        // dd($request->all());
        // Se veio via POST, busca o código do input
        if ($request->isMethod('post')) {
            $cod = $request->input('cod_uso');
            if (!$cod) {
                return redirect()->back()->with('error', 'Código não informado.');
            }
        }

        // Buscar o pedido com status aprovado
        $pedido = Pedido::where('registro', $cod)
            ->where('status', 'Aprovado')
            ->orderBy('id', 'DESC')
            ->firstOrFail();

        // Gerar QR Code se não existir
        $qrcodePath = public_path('storage/qrcode/' . $cod . '.png');
        if (!is_file($qrcodePath)) {
            QrCode::format('png')
                ->size(77)
                ->generate(route('validar.carteira', ['cod' => $cod]), $qrcodePath);
        }

        $qrcode = url('storage/qrcode/' . $cod . '.png');

        // Buscar o ingresso relacionado ao pedido
        $ingresso = Ingresso::where('pedido_id', $pedido->id)->firstOrFail();

        // Decodificar campos personalizados
        $campos = [];
        if (is_string($ingresso->campos_personalizados)) {
            $campos = json_decode($ingresso->campos_personalizados, true) ?? [];
        }

        // Preparar atributos do certificado
        $atributos = [
            'CN'           => $ingresso->nome_completo ?? 'Sem nome',
            'emailAddress' => $ingresso->email ?? '',
            'CPF'          => $ingresso->cpf ?? '',
            'Celular'      => $ingresso->celular ?? '',
            'Curso'        => $ingresso->curso ?? '',
            'Instituicao'  => $ingresso->instituicao_ensino,
            'Validade'     => $pedido->validade ?? '',
            'PedidoID'     => $pedido->id ?? '',
            'EventoID'     => $ingresso->evento_id ?? '',
            'Registro'     => $ingresso->registro ?? '',
        ];

        $certificado = $this->gerarCertificadoDeAtributos($atributos);
        $cliente = User::find($pedido->user_id);

        $view = ($pedido->evento_id == 155)
            ? 'carteira.carteira-v2-2026'
            : 'carteira.carteira-v2';

        return view($view, compact(
            'pedido',
            'cliente',
            'ingresso',
            'campos',
            'qrcode',
            'certificado'
        ));
    }

    public function gerarCertificadoDeAtributos(array $atributos)
    {
        $pathBase = storage_path('app/public/certificados');
        if (!is_dir($pathBase)) {
            mkdir($pathBase, 0777, true);
        }

        $certPath = $pathBase . '/certificado_atributo.pem';
        $keyPath = $pathBase . '/chave_privada.pem';

        // Gera um par de chaves RSA
        if (!file_exists($keyPath)) {
            $key = openssl_pkey_new([
                "private_key_bits" => 2048,
                "private_key_type" => OPENSSL_KEYTYPE_RSA,
            ]);
            openssl_pkey_export($key, $privateKey);
            file_put_contents($keyPath, $privateKey);
        } else {
            $privateKey = file_get_contents($keyPath);
            $key = openssl_pkey_get_private($privateKey);
        }

        // Configuração do certificado
        $dn = [
            "commonName" => $atributos['CN'] ?? 'Nome Padrão',
            "emailAddress" => $atributos['emailAddress'] ?? 'email@example.com',
            "organizationName" => "ICP-Brasil",
            "organizationalUnitName" => "UNIAO DOS ESTUDANTES DO BRASIL",
            "countryName" => "BR",
        ];

        // Criando uma requisição de certificado
        $csr = openssl_csr_new($dn, $key, ['digest_alg' => 'sha256']);
        $cert = openssl_csr_sign($csr, null, $key, 365, ['digest_alg' => 'sha256']);

        // Salvando o certificado
        openssl_x509_export($cert, $certOut);
        file_put_contents($certPath, $certOut);

        return $certOut;
    }
}
