<?php

namespace App\Http\Controllers;

use App\Models\Boleto;
use App\Models\GetWay;
use App\Models\Ingresso;
use App\Models\Log;
use App\Models\Pedido;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Milon\Barcode\DNS1D;
use Illuminate\Support\Facades\Http;
use Mail;
use App\Services\PagamentoService;

class IngressoController extends Controller
{


    protected function sendPushNotification($expoPushToken, $title, $body, $data = [])
    {
        $pushController = app(PushController::class);
        $request = new Request([
            'expoPushToken' => $expoPushToken,
            'title' => $title,
            'body' => $body,
            'data' => $data,
        ]);

        $pushController->sendNotification($request);
    }

    public function PedidoShow(Request $request)
    {
        $pedido = Pedido::find($request->id);
        if ($pedido->status == 'Pendente') {
            $this->RetornoMercadoPago($pedido->Boleto->id);
        }
        $pedido = Pedido::find($request->id);
        return view('cliente.ajax.pedido', ['pedido' => $pedido]);
    }


    public function RetornoMercadoPago($id)
    {
        // $id recebido é o ID do BOLETO
        $Boleto = Boleto::with(['Pedido', 'Pedido.Ingressos', 'Pedido.Evento'])->find($id);
        if (!$Boleto) {
            return response()->json([
                'success' => false,
                'message' => 'Boleto não encontrado para o ID informado.',
            ], 404);
        }

        // external_reference correto é o PEDIDO_ID que você enviou ao MP
        $pedidoId = $Boleto->pedido_id;
        $getway = GetWay::find($Boleto->getway_id);
        if (!$getway || empty($getway->access_token)) {
            return response()->json([
                'success' => false,
                'message' => 'GetWay não encontrado ou access_token ausente.',
            ], 500);
        }
        $accessToken = $getway->access_token;

        try {
            // Busca pelo external_reference (pedido_id)
            $response = Http::withToken($accessToken)
                ->timeout(20)
                ->get('https://api.mercadopago.com/v1/payments/search', [
                    'external_reference' => $Boleto->id,
                    'sort'               => 'date_created',
                    'criteria'           => 'desc',
                    'limit'              => 1,
                ]);

            if ($response->failed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Falha ao consultar pagamento.',
                    'status'  => $response->status(),
                    'body'    => $response->json(),
                ], $response->status());
            }

            $payload = $response->json();
            $results = $payload['results'] ?? [];

            if (empty($results)) {
                return response()->json([
                    'success' => true, // mantém o polling vivo no front
                    'response' => 'not_found',
                    'message' => 'Nenhum pagamento encontrado para o external_reference.',
                ], 200);
            }

            $dados = $results[0];
            $paymentId = (string) data_get($response->json(), 'id');

            // Campos úteis
            $payment_id        = $paymentId ?? null;
            $status            = $dados['status'] ?? null;               // approved, pending, in_process, rejected, cancelled...
            $status_detail     = $dados['status_detail'] ?? null;
            $payment_method_id = $dados['payment_method_id'] ?? null;    // pix, visa...
            $payment_type_id   = $dados['payment_type_id'] ?? null;      // bank_transfer (PIX), credit_card...
            $valor             = $dados['transaction_amount'] ?? null;
            $date_approved     = $dados['date_approved'] ?? null;

            // Atualiza o boleto com os dados do pagamento
            $Boleto->update([
                'payment_id'         => $paymentId,
                'status_mp'          => $status,
                'status_detail_mp'   => $status_detail,
                'payment_method_id'  => $payment_method_id,
                'payment_type_id'    => $payment_type_id,
                'valor_mp'           => $valor,
                'date_approved_mp'   => $date_approved,
                'external_reference' => $pedidoId,
            ]);

            // dd($Boleto);

            $Pedido = $Boleto->Pedido;
            if (!$Pedido) {
                return response()->json([
                    'success' => false,
                    'message' => 'Pedido não encontrado.',
                ], 404);
            }

            // Se aprovou, aplica regras de negócio e já devolve a URL de redirecionamento
            if ($status === 'approved') {

                $Boleto->update([
                    'status'    => '1',
                    'data_pago' => now()->toDateString(),
                ]);

                $Pedido->update(['status' => 'Aprovado']);

                $ultimoUsuario = null;
                $count = 0;

                foreach ($Pedido->Ingressos as $ingresso) {
                    $count++;
                    $ingresso->update(['status' => '1']);
                    // VERIFICANDO RULE OFF
                    if (!empty($ingresso->Evento->rule_off) && ($getway->tipo ?? null) === 'pf') {
                        $ingresso->update(['check_in_situacao' => '1']);
                        $Pedido->update(['check_in_situacao' => '1']);
                        foreach ($Pedido->Ingressos as $ddx) {
                            $ddx->update(['check_in_situacao' => '1']);
                        }
                    }

                    $user = User::find($ingresso->user_id);
                    $ultimoUsuario = $user;

                    if ($user) {
                        try {
                            Mail::send('emails.usuario-pdv', [
                                'name'   => $user->name,
                                'email'  => $user->email,
                                'evento' => $Pedido->Evento->titulo,
                                'banner' => $Pedido->Evento->banner,
                                'link'   => 'https://checkout.ucesub.com.br/print/ingressos/' . $Pedido->registro,
                                'qtd'    => $Pedido->Ingressos->count(),
                                'ref'    => $Pedido->Ingressos->count() . ' ingresso(s) ' . $Pedido->Evento->titulo,
                            ], function ($mail) use ($user) {
                                $mail->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                                $mail->to($user->email)->subject('Ingresso ' . env('APP_NAME'));
                            });
                        } catch (\Throwable $e) {
                            Log::warning('Falha ao enviar e-mail de ingresso', [
                                'pedido_id' => $Pedido->id,
                                'user_id'   => $user->id ?? null,
                                'erro'      => $e->getMessage(),
                            ]);
                        }
                    }
                }

                // Push notification (se existir)
                if ($ultimoUsuario && method_exists($this, 'sendPushNotification')) {
                    foreach ($ultimoUsuario->pushToken as $push) {
                        if ($push->token) {
                            $this->sendPushNotification(
                                $push->token,
                                'Ingresso Disponível',
                                'Seu ingresso para ' . $Pedido->Evento->titulo . ' já está disponível.',
                                ['evento_id' => $Pedido->Evento->id ?? 1]
                            );
                        }
                    }
                }

                return response()->json([
                    'success'   => true,
                    'response'  => 'approved',
                    'dados'     => $dados,
                    'getway'    => ['id' => $getway->id, 'tipo' => $getway->tipo],
                    'redirect'  => 'https://checkout.ucesub.com.br/success/' . $Pedido->id,
                ]);
            }

            // Para outros status, apenas informe o status atual
            return response()->json([
                'success'  => true,
                'response' => $status ?: 'unknown',
                'dados'    => $dados,
                'getway'   => ['id' => $getway->id, 'tipo' => $getway->tipo],
            ]);
        } catch (\Throwable $e) {
            Log::error('Erro ao consultar pagamento (external_reference)', [
                'boleto_id' => $id,
                'pedido_id' => $Boleto->pedido_id ?? null,
                'erro'      => $e->getMessage(),
                'line'      => $e->getLine(),
                'file'      => $e->getFile(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao consultar pagamento.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    public function PrintIngressos($registro)
    {
        $pedido = Pedido::where('registro', $registro)
            ->where('status', 'Aprovado')
            ->with('IngressosCliente.Pacote')
            ->first();

        if (empty($pedido)) {
            return view('site.404');
        }


        // 🔹 Gera QR Code e Código de Barras para todos os ingressos
        foreach ($pedido->IngressosCliente as $ingresso) {
            $ingresso->qr_code = QrCode::size(150)->generate($ingresso->registro);
            $barcode = new DNS1D();
            $barcode->setStorPath(__DIR__ . '/cache/');
            $ingresso->barcode_img = 'data:image/png;base64,' . $barcode->getBarcodePNG($ingresso->registro, 'C39');
        }

        return view('print.ingressos', ['pedido' => $pedido]);
    }





    /**
     * Display the specified resource.
     */
    public function chargeback($registro, Request $request)
    {
        $pedido = Pedido::where('registro', $registro)->with(['boleto', 'Cliente', 'Ingressos'])->firstOrFail();

        if ($request->acesso !== $pedido->boleto->payment_id) {
            abort(403, 'Acesso não autorizado.');
        }

        return view('print.recibo', compact('pedido'));
    }


    public function PedidoDestroy(Request $request)
    {
        // $pedido = Pedido::find($request->id);
        // if ($pedido->user_id != Auth::user()->id) {
        // } else {
        //     $pedido->delete();
        // }
    }

    public function EditarParticipantes($id)
    {
        $pedido = Pedido::where('id', $id)
            ->where('user_id', Auth::id())
            ->with('Ingressos')
            ->first();
        if ($pedido) {
            return view('cliente.editar-participantes', compact('pedido'));
        } else {
            return view('site.404');
        }
    }

    public function IngressoUpdate(Request $request)
    {
        foreach ($request->participantes as $participante) {
            $ingresso = Ingresso::find($participante['id']);
            $ingresso->update([
                'nome_completo' => $participante['nome'],
                'cpf'           => $participante['cpf'],
                'email'         => $participante['email'],
                'cpf'           => $participante['cpf'],
                'celular'       => $participante['celular'],
                'code_barras'   => gerarCodigo(12),
                'registro'      => gerarlETRARegistro(12),
                'update_nome'   => $ingresso->nome_completo != $participante['nome'] ? '1' : '0',
            ]);
        }

        return redirect()->route('cliente.ingressos')->with('success', 'Dados dos ingressos atualizados com sucesso');
    }
}
