<?php

namespace App\Http\Controllers;

use App\Models\Evento;
use App\Models\EventoColaborador;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DashBoardController extends Controller
{
    public function home()
    {
        $user = Auth::user();

        // Se for nível 2, limitar eventos aos permitidos
        $eventosQuery = Evento::with([
            'Repasses',
            'Pacotes.Ingressos' => function ($q) use ($user) {
                $q->where('status', '1')
                    ->where('check_in_situacao', '0')
                    ->with(['Pacote', 'Cupom', 'Pedido']);

                // 🔒 Filtro de segurança adicional para colaboradores (nível 2)
                if ($user->nivel == '2') {
                    $q->where('pdv_id', $user->produtor_id);
                }
            },
        ])
            ->where('status', 'Ativo')
            ->orderBy('data_inicio', 'ASC');

        // Se nível 2, limitar aos eventos com permissão
        if ($user->nivel == '2') {
            $eventosPermitidos = EventoColaborador::where('user_id', $user->id)->pluck('evento_id');
            $eventosQuery->whereIn('id', $eventosPermitidos);
        }

        $eventos = $eventosQuery->get();

        // Agora calcula o resumo de cada evento
        $eventos->transform(function ($evento) use ($user) {
            $ingressos = $evento->Pacotes->flatMap(fn($p) => $p->Ingressos);

            $vendasManuais = $ingressos->filter(fn($i) => $i->pdv === 'Manual')->sum('preco');
            $vendasSite    = $ingressos->filter(fn($i) => $i->pdv === 'Site')->sum('preco');

            $totalVendas = $vendasSite + $vendasManuais;
            $repasses = (float) $evento->Repasses->sum('valor');
            $saldo = $vendasSite - $repasses;

            $vendidos = 0;
            $capacidadeTotal = 0;

            foreach ($evento->Pacotes as $pacote) {
                $qtdVendidos = $pacote->Ingressos->filter(fn($i) => $i->pdv !== 'Manual')->count();
                $qtdPorPacote = $pacote->qtd_ingressos > 0 ? $pacote->qtd_ingressos : 1;
                $vendidos += $qtdVendidos / $qtdPorPacote;
                $capacidadeTotal += (int) ($pacote->qtd ?? 0);
            }

            $percentual = ($capacidadeTotal > 0)
                ? min(100, round(($vendidos / max(1, $capacidadeTotal)) * 100, 2))
                : 0;

            $evento->resumo = [
                'vendas_site'      => $vendasSite,
                'vendas_man'       => $vendasManuais,
                'total_vendas'     => $totalVendas,
                'repasses'         => $repasses,
                'saldo'            => $saldo,
                'vendidos'         => (int) $vendidos,
                'capacidade_total' => (int) $capacidadeTotal,
                'percentual'       => $percentual,
            ];

            return $evento;
        });

        return view('admin.home.index', compact('eventos'));
    }

    public function EventosInativos()
    {
        // Carrega tudo que precisamos evitando N+1
        $eventos = Evento::with([
            'Repasses',
            'Pacotes.Ingressos' => function ($q) {
                $q->where('status', '1')
                    ->where('check_in_situacao', '0')
                    ->with(['Pacote', 'Cupom', 'Pedido']); // já traz tudo o que precisamos
            },
        ])
            ->whereIN('status', ['Encerado', 'Pausado'])
            ->orderBy('data_inicio', 'DESC')
            ->get();

        $eventos->transform(function ($evento) {
            // ---- Coleção de ingressos válidos ----
            $ingressos = $evento->Pacotes->flatMap(fn($p) => $p->Ingressos);

            // Identifica origem (Manual/Site) priorizando pdv do ingresso
            $getPDV = function ($i) {
                if (!empty($i->pdv)) return $i->pdv;
                return optional($i->Pedido)->pdv ?? null;
            };

            // ---- Cálculo de vendas ----
            $vendasManuais = $ingressos
                ->filter(fn($i) => $getPDV($i) === 'Manual')
                ->sum(fn($i) => (float) ($i->preco ?? 0));

            $vendasSite = $ingressos
                ->filter(fn($i) => $getPDV($i) === 'Site')
                ->sum(fn($i) => (float) ($i->preco ?? 0));

            $totalVendas = $vendasSite + $vendasManuais;

            // Repasses e saldo (saldo considera só vendas site)
            $repasses = (float) $evento->Repasses->sum('valor');
            $saldo    = $vendasSite - $repasses;

            // ---- Barrinha de progresso ----
            $vendidos = 0;
            $capacidadeTotal = 0;

            foreach ($evento->Pacotes as $pacote) {
                $qtdVendidos = $pacote->Ingressos
                    ->filter(fn($i) => $getPDV($i) !== 'Manual')
                    ->count();

                // capacidade por pacote (se usar qtd_ingressos como limite)
                $qtdPorPacote = $pacote->qtd_ingressos > 0 ? $pacote->qtd_ingressos : 1;
                $vendidos += $qtdVendidos / $qtdPorPacote;

                $capacidadeTotal += (int) ($pacote->qtd ?? 0);
            }

            $percentual = ($capacidadeTotal > 0)
                ? min(100, round(($vendidos / max(1, $capacidadeTotal)) * 100, 2))
                : 0;

            // ---- Resumo final ----
            $evento->resumo = [
                'vendas_site'      => $vendasSite,
                'vendas_man'       => $vendasManuais,
                'total_vendas'     => $totalVendas,
                'repasses'         => $repasses,
                'saldo'            => $saldo,
                'vendidos'         => (int) $vendidos,
                'capacidade_total' => (int) $capacidadeTotal,
                'percentual'       => $percentual,
            ];

            return $evento;
        });

        return view('admin.home.index', compact('eventos'));
    }
}
