<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;
use Illuminate\Support\Facades\Validator;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        return view('auth.register');
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        // dd($request->all());
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'celular' => ['required', 'string', 'max:255'],
            'cpf' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $nomeSeparado = explode(' ', $request->name);
        $primeiroNome = $nomeSeparado[0];
        $sobreNome = implode(' ', array_slice($nomeSeparado, 1));

        $user = User::create([
            'name'          => $primeiroNome,
            'sobrenome'     => $sobreNome,
            'celular'       => $request->celular,
            'cpf'       => $request->cpf,
            'nivel'         => '3',
            'email'         => $request->email,
            'password'      => Hash::make($request->password),
        ]);


        event(new Registered($user));

        Auth::login($user);

        return redirect()->route('site.eventos');
    }


    public function StoreNew(Request $request): RedirectResponse
    {
        // Normaliza e mapeia "nome" -> "name" + "sobrenome"
        $fullName = trim((string) $request->input('nome', ''));
        [$firstName, $lastName] = $this->splitName($fullName);

        // Remove máscaras
        $cpf      = preg_replace('/\D/', '', (string) $request->input('cpf', ''));
        $celular  = preg_replace('/\D/', '', (string) $request->input('celular', ''));
        $cep      = preg_replace('/\D/', '', (string) $request->input('cep', ''));

        // Monta o array já saneado
        $data = [
            'nome'         => $fullName,
            'name'         => $firstName,
            'sobrenome'    => $lastName,
            'celular'      => $celular,
            'nascimento'   => $request->input('nascimento'),
            'cpf'          => $cpf,
            'rg'           => $request->input('rg'),
            'sexo'         => $request->input('sexo'),
            'cep'          => $cep,
            'endereco'     => $request->input('endereco'),
            'numero'       => $request->input('numero'),
            'bairro'       => $request->input('bairro'),
            'complemento'  => $request->input('complemento'),
            'cidade'       => $request->input('cidade'),
            'estado'       => strtoupper((string) $request->input('estado')),
            'email'        => strtolower((string) $request->input('email')),
            'password'     => $request->input('password'),
            'password_confirmation' => $request->input('password_confirmation'),
        ];

        // Validação (mensagens em PT-BR se quiser customizar)
        Validator::make(
            $data,
            [
                'nome'       => ['required', 'string', 'min:3', 'max:255'],
                'name'       => ['required', 'string', 'max:255'],
                'sobrenome'  => ['nullable', 'string', 'max:255'],

                // Celular: somente dígitos (10 ou 11) + único
                'celular'    => ['required', 'digits_between:10,11', 'unique:users,celular'],

                // CPF: 11 dígitos, único e com DV correto
                'cpf'        => [
                    'required',
                    'digits:11',
                    'unique:users,cpf',
                    function ($attr, $value, $fail) {
                        if (!$this->isValidCpf($value)) $fail('CPF inválido.');
                    }
                ],

                // E-mail
                'email'      => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:users,email'],

                // Demais campos
                'nascimento' => ['required', 'date', 'before:-16 years'], // ajuste a idade mínima se quiser
                'sexo'       => ['required', 'in:Masculino,Feminino'],
                'cep'        => ['required', 'digits:8'],
                'endereco'   => ['required', 'string', 'max:255'],
                'numero'     => ['nullable', 'string', 'max:20'],
                'bairro'     => ['required', 'string', 'max:255'],
                'cidade'     => ['required', 'string', 'max:255'],
                'estado'     => ['required', 'string', 'size:2'],

                // Senha
                'password'   => ['required', 'confirmed', Rules\Password::min(8)],
            ],
            [], // mensagens customizadas (opcional)
            ['nome' => 'nome completo'] // atributos (opcional)
        )->validate();

        // Cria o usuário (ajuste os campos existentes na sua tabela "users")
        $user = User::create([
            'name'          => $data['name'],
            'sobrenome'     => $data['sobrenome'],
            'celular'       => $data['celular'],
            'cpf'           => $data['cpf'],
            'sexo'          => $data['sexo'] ?? null,
            'nascimento'    => $data['nascimento'] ?? null,

            'cep'           => $data['cep'],
            'endereco'      => $data['endereco'],
            'numero'        => $data['numero'],
            'bairro'        => $data['bairro'],
            'complemento'   => $data['complemento'] ?? null,
            'cidade'        => $data['cidade'],
            'estado'        => $data['estado'],

            'nivel'         => 3, // se existir essa coluna/regra
            'email'         => $data['email'],
            'password'      => Hash::make($data['password']),
        ]);

        event(new Registered($user));
        Auth::login($user);

        return redirect()->route('site.eventos');
    }

    /** Divide "Nome Sobrenome" em [primeiro, resto] */
    private function splitName(string $full): array
    {
        $full = trim(preg_replace('/\s+/', ' ', $full));
        if ($full === '') return ['', ''];
        $parts = explode(' ', $full);
        $first = array_shift($parts);
        $last  = count($parts) ? implode(' ', $parts) : '';
        return [$first, $last];
    }

    /** Validação matemática do CPF */
    private function isValidCpf(string $cpf): bool
    {
        $cpf = preg_replace('/\D/', '', $cpf);
        if (strlen($cpf) !== 11 || preg_match('/^(\d)\1{10}$/', $cpf)) return false;

        for ($t = 9; $t < 11; $t++) {
            $d = 0;
            for ($c = 0; $c < $t; $c++) $d += $cpf[$c] * (($t + 1) - $c);
            $d = ((10 * $d) % 11) % 10;
            if ($cpf[$t] != $d) return false;
        }
        return true;
    }
}
